/* valaintegerliteral.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaintegerliteral.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valaexpression.h>




struct _ValaIntegerLiteralPrivate {
	char* _value;
};

#define VALA_INTEGER_LITERAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INTEGER_LITERAL, ValaIntegerLiteralPrivate))
enum  {
	VALA_INTEGER_LITERAL_DUMMY_PROPERTY,
	VALA_INTEGER_LITERAL_VALUE
};
static void vala_integer_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_integer_literal_real_to_string (ValaCodeNode* base);
static gboolean vala_integer_literal_real_is_pure (ValaExpression* base);
static gpointer vala_integer_literal_parent_class = NULL;
static void vala_integer_literal_dispose (GObject * obj);



/**
 * Creates a new integer literal.
 *
 * @param i      literal value
 * @param source reference to source code
 * @return       newly created integer literal
 */
ValaIntegerLiteral* vala_integer_literal_new (const char* i, ValaSourceReference* source) {
	ValaIntegerLiteral * self;
	g_return_val_if_fail (i != NULL, NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	self = g_object_newv (VALA_TYPE_INTEGER_LITERAL, 0, NULL);
	vala_integer_literal_set_value (self, i);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	return self;
}


static void vala_integer_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaIntegerLiteral * self;
	self = VALA_INTEGER_LITERAL (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_integer_literal (visitor, self);
	vala_code_visitor_visit_expression (visitor, VALA_EXPRESSION (self));
}


static char* vala_integer_literal_real_to_string (ValaCodeNode* base) {
	ValaIntegerLiteral * self;
	const char* _tmp0;
	self = VALA_INTEGER_LITERAL (base);
	_tmp0 = NULL;
	return (_tmp0 = self->priv->_value, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
}


/**
 * Returns the type name of the value this literal represents.
 *
 * @return the name of literal type
 */
char* vala_integer_literal_get_type_name (ValaIntegerLiteral* self) {
	const char* _tmp0;
	char* number;
	gint l;
	gboolean u;
	gint64 n;
	g_return_val_if_fail (VALA_IS_INTEGER_LITERAL (self), NULL);
	_tmp0 = NULL;
	number = (_tmp0 = self->priv->_value, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	l = 0;
	while (g_str_has_suffix (number, "l") || g_str_has_suffix (number, "L")) {
		char* _tmp1;
		l++;
		_tmp1 = NULL;
		number = (_tmp1 = g_strndup (number, ((gulong) (strlen (number) - 1))), (number = (g_free (number), NULL)), _tmp1);
	}
	u = FALSE;
	if (g_str_has_suffix (number, "u") || g_str_has_suffix (number, "U")) {
		char* _tmp2;
		u = TRUE;
		_tmp2 = NULL;
		number = (_tmp2 = g_strndup (number, ((gulong) (strlen (number) - 1))), (number = (g_free (number), NULL)), _tmp2);
	}
	n = g_ascii_strtoll (number, NULL, 0);
	if (!u && n > 0x7fffffff) {
		/* value doesn't fit into signed 32-bit*/
		l = 2;
	} else {
		if (u && n > 0xffffffff) {
			/* value doesn't fit into unsigned 32-bit*/
			l = 2;
		}
	}
	if (l == 0) {
		if (u) {
			char* _tmp3;
			_tmp3 = NULL;
			return (_tmp3 = g_strdup ("uint"), (number = (g_free (number), NULL)), _tmp3);
		} else {
			char* _tmp4;
			_tmp4 = NULL;
			return (_tmp4 = g_strdup ("int"), (number = (g_free (number), NULL)), _tmp4);
		}
	} else {
		if (l == 1) {
			if (u) {
				char* _tmp5;
				_tmp5 = NULL;
				return (_tmp5 = g_strdup ("ulong"), (number = (g_free (number), NULL)), _tmp5);
			} else {
				char* _tmp6;
				_tmp6 = NULL;
				return (_tmp6 = g_strdup ("long"), (number = (g_free (number), NULL)), _tmp6);
			}
		} else {
			if (u) {
				char* _tmp7;
				_tmp7 = NULL;
				return (_tmp7 = g_strdup ("uint64"), (number = (g_free (number), NULL)), _tmp7);
			} else {
				char* _tmp8;
				_tmp8 = NULL;
				return (_tmp8 = g_strdup ("int64"), (number = (g_free (number), NULL)), _tmp8);
			}
		}
	}
	number = (g_free (number), NULL);
}


static gboolean vala_integer_literal_real_is_pure (ValaExpression* base) {
	ValaIntegerLiteral * self;
	self = VALA_INTEGER_LITERAL (base);
	return TRUE;
}


const char* vala_integer_literal_get_value (ValaIntegerLiteral* self) {
	g_return_val_if_fail (VALA_IS_INTEGER_LITERAL (self), NULL);
	return self->priv->_value;
}


void vala_integer_literal_set_value (ValaIntegerLiteral* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_INTEGER_LITERAL (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_value = (g_free (self->priv->_value), NULL)), _tmp2);
}


static void vala_integer_literal_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaIntegerLiteral * self;
	self = VALA_INTEGER_LITERAL (object);
	switch (property_id) {
		case VALA_INTEGER_LITERAL_VALUE:
		g_value_set_string (value, vala_integer_literal_get_value (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_integer_literal_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaIntegerLiteral * self;
	self = VALA_INTEGER_LITERAL (object);
	switch (property_id) {
		case VALA_INTEGER_LITERAL_VALUE:
		vala_integer_literal_set_value (self, g_value_get_string (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_integer_literal_class_init (ValaIntegerLiteralClass * klass) {
	vala_integer_literal_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaIntegerLiteralPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_integer_literal_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_integer_literal_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_integer_literal_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_integer_literal_real_accept;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_integer_literal_real_to_string;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_integer_literal_real_is_pure;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_INTEGER_LITERAL_VALUE, g_param_spec_string ("value", "value", "value", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_integer_literal_instance_init (ValaIntegerLiteral * self) {
	self->priv = VALA_INTEGER_LITERAL_GET_PRIVATE (self);
}


static void vala_integer_literal_dispose (GObject * obj) {
	ValaIntegerLiteral * self;
	self = VALA_INTEGER_LITERAL (obj);
	self->priv->_value = (g_free (self->priv->_value), NULL);
	G_OBJECT_CLASS (vala_integer_literal_parent_class)->dispose (obj);
}


GType vala_integer_literal_get_type (void) {
	static GType vala_integer_literal_type_id = 0;
	if (G_UNLIKELY (vala_integer_literal_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaIntegerLiteralClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_integer_literal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaIntegerLiteral), 0, (GInstanceInitFunc) vala_integer_literal_instance_init };
		vala_integer_literal_type_id = g_type_register_static (VALA_TYPE_LITERAL, "ValaIntegerLiteral", &g_define_type_info, 0);
	}
	return vala_integer_literal_type_id;
}




