#!/usr/bin/env python

#
# Pyne - Python Newsreader and Emailer
#
# Copyright (c) 2000, 2001 Tom Morton
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#        
# Tom Morton <tom@moretom.net>
#

import utils
from gtk import *
import GtkExtra
import cPickle
import time
import os
import os.path
import sys
import string
from copy import copy

from addressbook import *
from ptk.big_edit_box import *

import pynei18n
import mainwin
from pyneheaders import *
import pynemsg
from boxtypes import *

# This is sent with every message. X-Mailer:
ver_string = "Pyne 0.6.7 (%s)"
ver_stamp = (0,6,7)

HPANE_POS = 2
VPANE_POS = 3

# mark messages as read only when double-clicked
OPT_2CLICK_MARK   = 0x00000001
# 'classic' pane configuration
OPT_PANES_CLASSIC = 0x00000002

# remember location of pyne modules
pyne_path = os.path.split( os.path.abspath(sys.argv[0]) )[0]

# complete the pyne version string: host OS
os_info = os.popen("uname -s").read() # uname -sr probably reveals too much
if os_info == "": os_info = "Unknown OS"
ver_string = ver_string % string.strip(os_info)


class pyne_user:
	"""
	Each user of pyne will have one of these objects.
	They may contain mailboxes, outboxes, NNTPboxes, etc.
	"""

	def __init__(self):
		"""
		Create a new user.
		Give them an inbox, outbox,
		UNFINISHED: deleted, sent and saved folders.
		"""
		self.contents = []

		# version stamp
		self.ver_stamp = ver_stamp

		# Window size info
		# (width, height, hpane position, vpane position)
		self.window_setup = []

		# Font used for message body text
		self.bodyfont = "-*-fixed-*-*-*--*-*-*-*-*-*-*-9"
		self.linewrap = 72
		self.replyhead = "On $DATE, $FROM wrote:"

		# Create new boxes
		# 'special' uids for non-deletable special folders (outbox,
		# sent, deleted)
		a = outbox.outbox(self, "Outbox", "outbox")
		self.contents.append(a)

		a = storebox.storebox(self, "Drafts", self.get_uid())
		self.contents.append(a)

		a = storebox.storebox(self, "Sent", "sent")
		self.contents.append(a)
		
		a = storebox.storebox(self, "Saved", self.get_uid())
		self.contents.append(a)
		
		a = deletedbox.deletedbox(self, "Deleted", "deleted")
		self.contents.append(a)

		# Address book
		self.addressbook = AddressBook()

		# List of open windows
		self.windows = {}

		# List of to-be-deleted temporary files
		self.tempfiles = []

		self.col_text = [ 0.0, 0.0, 0.0 ]
		self.col_quote = [ 0.0, 0.0, 0.8 ]
		self.col_header = [ 0.8, 0.0, 0.0 ]

		# default: sort messages by date, newest to top (2==date)
		self.sort_type = (2, 0)

		# various boolean options. see OPT_xx at top
		self.opts = 0

		# printer command
		self.printer = 'lpr'

		# When we last expired stuff
		self.last_expiry = time.gmtime(time.time())[:3]

		# position of tabs in quickview and composer
		self.tab_pos = POS_TOP

		# parse html bodies with this:
		self.html_parser = "lynx -dump"

	def recover(self, path):
		"""
		Not appropriate to pyne mailboxes...
		"""
		pass

	def parent_of(self, folder):
		"""
		Return the parent object of 'folder'.
		"""
		uid = folder.uid
		def find_parent(folder, uid=uid):
			if folder.__dict__.has_key("contents"):
				for x in folder.contents:
					if x.uid == uid:
						return folder
		return utils.recurse_apply( [self], find_parent)[0]

	def get_folder_by_uid(self, uid):
		"""
		Return folder in user.contents with uid==uid :-)
		"""
		# Clean up shit-brained usage of integer and string UIDS
		### deprecate int uids some version XXX
		try:
			uid = int(uid)
		except ValueError:
			pass

		def get_uid_matches(folder, uid=uid):
			if folder.uid == uid:
				return folder
		folders = utils.recurse_apply(self.contents, get_uid_matches)
		if len(folders) == 0:
			return None
		else:
			return folders[0]

	def set_preferences(self):
		"""
		Allow the user to set his preferences like fonts, etc.
		"""
		######
		#import ptk.misc_widgets
		## this whole thing is such a mess and must be moved and cleaned up
		win = GtkWindow()
		win.set_title(_("Pyne Preferences"))

		box1 = GtkVBox()
		win.add(box1)
		box1.show()

		notebook = GtkNotebook()
		notebook.set_tab_pos(POS_TOP)
		box1.pack_start(notebook)
		notebook.show()

		######### Messages options page
		frame = GtkFrame(_(" Misc "))
		frame.set_border_width(5)
		frame.set_usize(500,300)
		frame.show()

		miscbox = GtkVBox()
		frame.add(miscbox)
		miscbox.show()

		bdfnt_box = GtkHBox(spacing=5)
		bdfnt_box.set_border_width(5)
		miscbox.pack_start(bdfnt_box, expand=FALSE)
		bdfnt_box.show()
		label = GtkLabel(_("Message body font:"))
		bdfnt_box.pack_start(label, expand=FALSE)
		label.show()

		bodyfont = GtkEntry()
		bodyfont.set_text(str(self.bodyfont))
		bdfnt_box.pack_start(bodyfont)
		bodyfont.show()

		def font_box(_button, bodyfont=bodyfont):
			f = GtkFontSelectionDialog(_("Message Body Font"))
			f.set_font_name(bodyfont.get_text())
			def setfont(_button, f=f, bodyfont=bodyfont):
				bodyfont.set_text(f.get_font_name())
				f.destroy()
			f.show()
			f.ok_button.connect("clicked", setfont)
			f.cancel_button.connect("clicked", f.destroy)

		button = GtkButton(_(" Select "))
		button.connect("clicked", font_box)
		bdfnt_box.pack_start(button, expand=FALSE)
		button.show()

		settings_box = big_edit_box( self,
		      ( ("html_parser", _("Parse html bodies with:"), VAR_TYPE_STRING, 0, 0),
		        ("linewrap", _("Wrap lines at:"), VAR_TYPE_INTEGER, 0, 72),
		        ("replyhead", _("Reply header:"), VAR_TYPE_STRING, 0, 0),
			("opts", _("Double click to mark messages as read"), VAR_TYPE_PACKED_BIT_BOOLEAN, 0, OPT_2CLICK_MARK),
			("opts", _("Classic window layout"), VAR_TYPE_PACKED_BIT_BOOLEAN, 0, OPT_PANES_CLASSIC) )
		)
		miscbox.pack_start(settings_box, expand=FALSE)
		settings_box.show()

		tabpos_box = GtkHBox(spacing=5)
		miscbox.pack_start(tabpos_box, expand=FALSE)
		tabpos_box.show()
		
		menu = GtkMenu()
		menu.show()
		new_tabpos = [ self.tab_pos ]

		def ch_tabpos(item, new_tabpos=new_tabpos):
			new_tabpos[0] = item.get_data("0")

		items = [ _("Left"), _("Right"), _("Top"), _("Bottom") ]
		for x in range(0, len(items)):
			menuitem = GtkMenuItem(items[x])
			menuitem.set_data("0", x)
			menuitem.connect("activate", ch_tabpos)
			menu.append(menuitem)
			menuitem.show()
		optmenu = GtkOptionMenu()
		optmenu.set_menu(menu)
		tabpos_box.pack_end(optmenu, expand=FALSE)
		optmenu.set_history(self.tab_pos)
		optmenu.show()

		label = GtkLabel(_("Message view tab position:"))
		tabpos_box.pack_end(label, expand=FALSE)
		label.show()
		
		label = GtkLabel(_("Messages"))
		notebook.append_page(frame, label)

		######### Text colours page
		frame = GtkFrame(_(" Colours "))
		frame.set_border_width(10)
		frame.set_usize(400,200)
		frame.show()
		label = GtkLabel(_("Colours"))
		notebook.append_page(frame, label)

		box3 = GtkVBox()
		box3.set_border_width(5)
		frame.add(box3)
		box3.show()

		box3_1 = GtkHBox(spacing=5)
		box3_1.set_border_width(5)
		box3.pack_start(box3_1, expand=FALSE)
		box3_1.show()
		
		col_text = copy(self.col_text)
		col_header = copy(self.col_header)
		col_quote = copy(self.col_quote)

		c = GtkColorSelection()
		c.change_col = None
		def _change(this, c=c, ct=col_text, ch=col_header, cq=col_quote):
			if c.change_col == 0:
				col = ct
			elif c.change_col == 1:
				col = ch
			elif c.change_col == 2:
				col = cq
			else:
				return
			del col[0]
			del col[0]
			del col[0]
			x = this.get_color()
			col.append(x[0])
			col.append(x[1])
			col.append(x[2])
		c.connect("color-changed", _change)
		box3.pack_start(c)
		c.show()

		def _bt(_button, c=c, col=col_text):
			c.change_col = 0
			c.set_color((col[0], col[1], col[2], 1.0))

		def _qt(_button, c=c, col=col_quote):
			c.change_col = 2
			c.set_color((col[0], col[1], col[2], 1.0))

		def _ht(_button, c=c, col=col_header):
			c.change_col = 1
			c.set_color((col[0], col[1], col[2], 1.0))

		button = GtkButton(_(" Body Text "))
		button.connect("clicked", _bt)
		box3_1.pack_start(button, expand=FALSE)
		button.show()
		button = GtkButton(_(" Quoted Text "))
		button.connect("clicked", _qt)
		box3_1.pack_start(button, expand=FALSE)
		button.show()
		button = GtkButton(_(" Headers "))
		button.connect("clicked", _ht)
		box3_1.pack_start(button, expand=FALSE)
		button.show()

		##########################
		box2 = GtkHBox(spacing=5)
		box2.set_border_width(5)
		box1.pack_start(box2, expand=FALSE)
		box2.show()

		button = GtkButton(_(" Cancel "))
		button.connect("clicked", win.destroy)
		box2.pack_end(button, expand=FALSE)
		button.show()

		def _set_prefs(_button, win=win, self=self, settings_box=settings_box, bodyfont=bodyfont,
				col_text=col_text, col_header=col_header,
				col_quote=col_quote, new_tabpos=new_tabpos):
			"""
			Save the changed preferences.
			"""
			# shite
			self.col_text = col_text
			self.col_header = col_header
			self.col_quote = col_quote

			self.bodyfont = bodyfont.get_text()
			settings_box.apply_changes()
			if self.tab_pos != new_tabpos[0]:
				# change quickview tab positions
				self.tab_pos = new_tabpos[0]
				for i in self.windows.keys():
					self.windows[i].msg_view.set_tab_pos(self.tab_pos)
			
			win.destroy()

		button = GtkButton(" "+_("Ok")+" ")#ptk.misc_widgets.OkButton()
		button.connect("clicked", _set_prefs)
		box2.pack_end(button, expand=FALSE)
		button.show()

		win.show()


	def update(self, update_type=0):
		"""
		Update all windows' folder lists.

		By default update changed stuff only.
		"""
		for x in self.windows.keys():
			self.windows[x].update(self.contents, update_type)
		# removed changed markers
		def remove_changed(folder):
			if folder.__dict__.has_key("changed"):
				del folder.changed
		# from 'utils.py'
		utils.recurse_apply(self.contents, remove_changed)


	def expire(self):
		"""
		Expire messages collected longer ago than self.expire_after
		days.
		"""
		def expire_msgs(expire_msgs, folder, this_day, expire_after):
			"""
			Recursively remove links to msg_id in object.
			"""
			# if the object contains messages, remove msg_ids from
			# them
			if folder.__dict__.has_key("messages"):
				messages = copy(folder.messages)
				for x in messages:
					msg = folder.io.load_header(x)
					# fucked up headers
					try:
						len(msg)
					except TypeError, e:
						print "*",
						folder.io.delete_article(x)
						try:
							folder.messages.remove(x)
						except ValueError, e:
							pass
						continue
					# test age
					date_received = msg[HEAD_DATE_RECEIVED]
					day = date_received[0]*365 + date_received[7]
					old = this_day - day
					if old >= expire_after:
						# delete it
						folder.io.delete_article(x)
						try:
							folder.messages.remove(x)
						except ValueError, e:
							pass

		############################################################
		this_date = time.gmtime(time.time())
		this_day = this_date[0]*365 + this_date[7]

		# only test for expiry in folders with expire_after
		def _pre_expire(folder, expire_msgs=expire_msgs, this_day=this_day):
			if not folder.__dict__.has_key("expire_after"):
				return
			if folder.expire_after == None:
				return
			print "Pyne: Expiring folder ", folder.name
			# recurse into folder looking for messages that have
			# expired.
			expire_msgs(expire_msgs, folder, this_day, folder.expire_after)
		utils.recurse_apply(self.contents, _pre_expire)
					
	def kill_window(self, num):
		self.windows[num].win.destroy()
		del self.windows[num]
		# All windows closed: quit
		if len(self.windows) == 0:
			mainquit()
		return

	def new_window(self, display_msg = None):
		"""
		Open new window, with maximised quickview pane and showing
		message (folder, msg-id) 'display_msg' if it is != None.
		"""
		# Find unused windows number
		x = 0
		while self.windows.has_key(x):
			x = x + 1
		if len(self.window_setup) <= x:
			self.window_setup.append( (600, 400, -1, -1) )
		win = mainwin.pyne_window(self, ver_string, self.window_setup[x], x, display_msg = display_msg)
		win.update(self.contents, mainwin.UPDATE_FOLDERVIEW)
		self.windows[x] = win

	def get_uid(self):
		"""
		Return unique id for new object.
		"""
		# build list of current ids
		def get_uids(folder):
			return folder.uid
		uids = utils.recurse_apply(self.contents, get_uids)
		# return an unused id
		uid = 0
		while 1:
			if not uid in uids:
				return uid
			uid = uid + 1

	def start(self):
		"""
		Just get on with it...
		"""
		# Load message pixmaps
		# Key format "%d%d%d" % (cached, isread, isreplied, ismarked)
		self.msg_icons = {}

		# just to shut it up with all the damn warnings we make this...
		# shame on me :-(
		SHITWIN = GtkWindow()
		pix, mask = create_pixmap_from_xpm(SHITWIN, None, pyne_path+"/icons/msg_read.xpm")
		self.msg_icons["1100"] = (pix, mask)
		pix, mask = create_pixmap_from_xpm(SHITWIN, None, pyne_path+"/icons/msg_unread.xpm")
		self.msg_icons["1000"] = (pix, mask)
		pix, mask = create_pixmap_from_xpm(SHITWIN, None, pyne_path+"/icons/msg_read_replied.xpm")
		self.msg_icons["1110"] = (pix, mask)
		pix, mask = create_pixmap_from_xpm(SHITWIN, None, pyne_path+"/icons/msg_uncached.xpm")
		self.msg_icons["0000"] = (pix, mask)
		pix, mask = create_pixmap_from_xpm(SHITWIN, None, pyne_path+"/icons/msg_unread_replied.xpm")
		self.msg_icons["1010"] = (pix, mask)
		pix, mask = create_pixmap_from_xpm(SHITWIN, None, pyne_path+"/icons/msg_read_marked.xpm")
		self.msg_icons["1101"] = (pix, mask)
		pix, mask = create_pixmap_from_xpm(SHITWIN, None, pyne_path+"/icons/msg_unread_marked.xpm")
		self.msg_icons["1001"] = (pix, mask)
		pix, mask = create_pixmap_from_xpm(SHITWIN, None, pyne_path+"/icons/msg_read_replied_marked.xpm")
		self.msg_icons["1111"] = (pix, mask)
		pix, mask = create_pixmap_from_xpm(SHITWIN, None, pyne_path+"/icons/msg_uncached_marked.xpm")
		self.msg_icons["0001"] = (pix, mask)
		pix, mask = create_pixmap_from_xpm(SHITWIN, None, pyne_path+"/icons/msg_unread_replied_marked.xpm")
		self.msg_icons["1011"] = (pix, mask)

		# Start up main window
		self.new_window()

		# Input loop
		mainloop()
	
		# Trash temporary files
		for x in self.tempfiles:
			# delete temporary files
			try:
				os.remove(x)
			except OSError:
				pass
		self.tempfiles = []

		# expire if not done today
		if self.last_expiry != time.gmtime(time.time())[:3]:
			self.last_expiry = time.gmtime(time.time())[:3]
			self.expire()
	
		# remove stuff we don't want to save
		del self.msg_icons

		# Save datafile
		saveuser(self)

		# End
		print "Pyne exited."


def saveuser(user):
	"""
	Save ~/.pyne/user.dat
	"""
	def _cleanup(folder):
		try:
			folder.clean_4_save()
		except AttributeError, e:
			pass
	utils.recurse_apply(user.contents, _cleanup)

	if os.path.isfile("user.dat"):
		# keep a backup copy of the last user.dat file
		os.rename("user.dat", "user.bak")
	f = open("user.dat", "w")
	p = cPickle.Pickler(f, 1)
	
	p.dump(user)

	f.close()


def loaduser():
	"""
	Load ~/.pyne/user.dat
	"""
	f = open("user.dat", "r")
	u = cPickle.Unpickler(f)

	try:
		object = u.load()
	except cPickle.UnpicklingError:
		f.close()
		print "Error: user.dat corrupt or old. Trying backups..."
		# try the backup. fail gracelessly this time
		if os.path.isfile("user.bak"):
			os.rename("user.bak", "user.dat")
			f = open("user.dat", "r")
			u = cPickle.Unpickler(f)
			object = u.load()
			print "Backups OK :-)"

	f.close()

	# check for very old user data:
	if not object.__dict__.has_key("ver_stamp"):
		print "Pyne 0.5.0 or older user data. Run convert.py to convert to ", ver_string
		os.remove("pyne.lock")
		sys.exit()

	# old user data
	if object.ver_stamp == (0,5,1):
		# add missing stuff
		object.ver_stamp = (0,5,2)
		object.printer = 'lpr'

	if object.ver_stamp < (0,5,11):
		# must have 'thread' and 'collection_threads' on newsgroups and
		# nntpboxes respectively
		object.ver_stamp = (0,5,11)
		def _fix_0_5_11(folder):
			if isinstance(folder, nntpbox.newsgroup):
				folder.thread = 1
			if isinstance(folder, nntpbox.nntpbox):
				folder.collection_threads = 4
		utils.recurse_apply(object.contents, _fix_0_5_11)

	if object.ver_stamp < (0,5,12):
		# need 'contents' list for popboxes (sub-folder support)
		# need 'expire_after' for newsgroups
		object.ver_stamp = (0,5,12)
		def _fix_0_5_12(folder, object=object):
			if isinstance(folder, popbox.popbox):
				folder.contents = []
			if isinstance(folder, nntpbox.newsgroup):
				parent_nntpbox = object.parent_of(folder)
				folder.expire_after = parent_nntpbox.expire_after
		utils.recurse_apply(object.contents, _fix_0_5_12)

	if object.ver_stamp < (0,5,13):
		# no longer need 'thread' boolean on message boxes to be
		# threaded. new 'opts' thingy.
		# message headers have isread and isreplied in one packed
		# bit 32 bit integer thingy.
		object.ver_stamp = (0,5,13)
		def _fix_0_5_13(folder, object=object):
			folder.opts = 0
			if folder.__dict__.has_key("thread"):
				folder.opts = folder.opts | superbox.OPT_THREAD
				del folder.thread
			if not isinstance(folder, nntpbox.newsgroup):
				folder.startup(object)
				print "Converting messages in", folder.name
				for i in folder.io.get_contents():
					try:
						header, body = folder.io._load_article(i)
					except KeyError, e:
						continue
					if header == None:
						continue
					else:
						msg = pynemsg.pynemsg()
						msg.opts = 0
						msg.body = body
						msg.parseheaders(object)
						msg.date = header[0]
						msg.date_received = header[1]
						if header[7]:
							msg.opts = msg.opts | MSG_ISREAD
						if header[8]:
							msg.opts = msg.opts | MSG_ISREPLIED
						msg.senduid = header[9]
						
						folder.io.save_article(msg)
				folder.clean_4_save()
		utils.recurse_apply(object.contents, _fix_0_5_13)

	if object.ver_stamp < (0,5,14):
		# news group lists now stored as string rather than bulky tuple
		object.ver_stamp = (0,5,14)
		def _fix_0_5_14(folder, object=object):
			if isinstance(folder, nntpbox.nntpbox):
				groups = folder.groups
				folder.groups = []
				for i in groups:
					folder.groups.append("%s %s" % i)
		utils.recurse_apply(object.contents, _fix_0_5_14)

	if object.ver_stamp < (0,5,15):
		# new: some boolean options
		object.ver_stamp = (0,5,15)
		object.opts = 0
		if object.pane_style == 1:
			object.opts = OPT_PANES_CLASSIC
		del object.pane_style

	if object.ver_stamp < (0,5,16):
		# reply header thingy and newsgroups now have filters, not nntpbox
		object.ver_stamp = (0,5,16)
		def _fix_0_5_16(folder, object=object):
			if isinstance(folder, nntpbox.nntpbox):
				del folder.filters
			elif isinstance(folder, nntpbox.newsgroup):
				folder.filters = []
		utils.recurse_apply(object.contents, _fix_0_5_16)
		object.replyhead = "On $DATE, $FROM wrote:"

	if object.ver_stamp < (0,5,17):
		# popboxes obsoleted. we now use mailboxes.
		object.ver_stamp = (0,5,17)
		def _fix_0_5_17(folder, object=object):
			if not folder.__dict__.has_key("contents"):
				return
			new_contents = []
			for i in folder.contents:
				if not isinstance(i, popbox.popbox):
					new_contents.append(i)
					continue
				# otherwise, convert the popbox. woo
				print "Converting ", i.name, " to mailbox format"
				new_box = mailbox.mailbox(object, i.name, i.uid, no_startup=1)
				if i.server == "":
					new_box.recv_type = 0 # RECV_NONE
				if i.smtpserver == "":
					new_box.send_type = 0 # SEND_NONE
				new_box.recv_conf["server"] = i.server
				new_box.recv_conf["port"] = i.port
				new_box.recv_conf["password"] = i.password
				new_box.recv_conf["username"] = i.username
				new_box.send_conf["server"] = i.smtpserver
				new_box.send_conf["port"] = i.smtpport
				new_box.realname = i.realname
				new_box.emailaddr = i.emailaddr
				new_box.replyto = i.replyto
				new_box.org = i.org
				new_box.sigfile = i.sigfile
				new_box.expire_after = i.expire_after
				new_box.export_format = i.export_format
				new_box.contents = i.contents
				new_box.filters = i.filters
				new_box.opts = i.opts

				new_contents.append(new_box)
			folder.contents = new_contents
		utils.recurse_apply([ object ], _fix_0_5_17)
		# need last expiry
		object.last_expiry = time.gmtime(time.time())[:3]

	if object.ver_stamp < (0,5,20):
		# composer and quickview tab positions
		object.tab_pos = POS_TOP
		object.ver_stamp = (0,5,20)

	if object.ver_stamp < (0,5,21):
		# we now can parse html bodies
		object.html_parser = "lynx -dump"
		object.ver_stamp = (0,5,21)

	if object.ver_stamp < (0,5,22):
		# need 'contents' list for storeboxes (sub-folder support)
		object.ver_stamp = (0,5,22)
		def _fix_0_5_22(folder, object=object):
			if isinstance(folder, storebox.storebox):
				folder.contents = []
		utils.recurse_apply(object.contents, _fix_0_5_22)
	
	if object.ver_stamp < (0,6,5):
		# Sent messages now need (or atleast should) be marked sent
		object.ver_stamp = (0,6,5)
		def _fix_0_6_5(folder, object=object):
			if folder.uid != "sent":
				return
			# Start folder so we can touch it up...
			folder.startup(object)
			for msg_id in folder.messages:
				msg = folder.io.load_article(msg_id)
				msg.opts = msg.opts | MSG_SENT
				folder.io.save_article(msg)
			folder.clean_4_save()
		utils.recurse_apply(object.contents, _fix_0_6_5)

	# open mailboxes
	def _start_boxes(folder, object=object):
		if folder.__dict__.has_key("export_format"):
			folder.startup(object)
	utils.recurse_apply(object.contents, _start_boxes)
	return object
	
if __name__ == '__main__':
	# Run and silently exit. Used by the installer to compile
	# the modules
	if sys.argv[-1] == "--compile":
		sys.exit(0)
	# Help
	if sys.argv[-1] == "--help":
		print "Usage: pyne [option] [user location]"
		print "Pyne is a GTK+ Newsreader/Emailer written in Python."
		print
		print "User location is optional, and defaults to ~/.pyne"
		print
		print "  -f, --force      Force startup after a crash by removing the lockfile"
		print
		print "Report bugs to <tom@moretom.net>"
		sys.exit(0)
	# get alternative location of .pyne:
	for arg in sys.argv[1:]:
		# skip other args
		if arg[0] == "-":
			continue
		user_home = sys.argv[1]
		break
	else:
		user_home = os.path.join(os.environ["HOME"], ".pyne")
	# Remember this as location of pyne modules
	sys.path.append(pyne_path)
	# Change to the working directory
	try:
		os.chdir(user_home)
	except OSError, e:
		# It's either non-existant, or not a directory (eek!)
		try:
			os.mkdir(user_home)
			os.chdir(user_home)
		except OSError, e:
			print "Error. Cannot create ~/%s/, or file exists with that name. HELP!!! :~{" % user_home
			sys.exit(0)
	# other arguments
	for arg in sys.argv[1:]:
		if arg == "-f" or arg == "--force":
			try:
				os.remove("pyne.lock")
			except OSError, e: pass
	# Check for a lock file
	try:
		f = open("pyne.lock", "r")
	except IOError, e:
		# None. make one.
		f = open("pyne.lock", "w")
		f.close()
	else:
		if GtkExtra.message_box("Warning!",
"There is already an instance of Pyne running or a Pyne\n\
session terminated abnormally (crashed :-)\n\
If so delete the %s/pyne.lock file." % user_home, ("Quit", "Ignore")) != "Ignore":
			sys.exit()
	# Attempt to open datafile
	try:
		usr = loaduser()
	except IOError, e:
		# Otherwise start new user
		usr = pyne_user()
	# start main thread: usr.start() 
	usr.start()

	# remove the lock file
	os.remove("pyne.lock")

