#!/usr/bin/env ruby

$:.unshift("../lib").unshift("../../lib") if __FILE__ =~ /\.rb$/

require 'puppet'
require 'puppettest'

class TestStorage < Test::Unit::TestCase
	include PuppetTest

    def mkfile
        path = tempfile()
        File.open(path, "w") { |f| f.puts :yayness }

        f = Puppet.type(:file).create(
            :name => path,
            :check => %w{checksum type}
        )

        return f
    end

    def test_storeandretrieve
        path = tempfile()

        f = mkfile()

        # Load first, since that's what we do in the code base; this creates
        # all of the necessary directories.
        assert_nothing_raised {
            Puppet::Storage.load
        }

        hash = {:a => :b, :c => :d}

        state = nil
        assert_nothing_raised {
            state = Puppet::Storage.cache(f)
        }

        assert(!state.include?("name"))

        assert_nothing_raised {
            state["name"] = hash
        }

        assert_nothing_raised {
            Puppet::Storage.store
        }
        assert_nothing_raised {
            Puppet::Storage.clear
        }
        assert_nothing_raised {
            Puppet::Storage.load
        }

        # Reset it
        state = nil
        assert_nothing_raised {
            state = Puppet::Storage.cache(f)
        }

        assert_equal(state["name"], hash)
    end

    # we're getting corrupt files, probably because multiple processes
    # are reading or writing the file at once
    # so we need to test that
    def test_multiwrite
        f = mkfile()

        value = {:a => :b}
        threads = []
        9.times { |a|
            threads << Thread.new {
                9.times { |b|
                    assert_nothing_raised {
                        Puppet::Storage.load
                        state = Puppet::Storage.cache(f)
                        value.each { |k,v| state[k] = v }
                        state[:e] = rand(100)
                        Puppet::Storage.store
                    }
                }
            }
        }
        threads.each { |th| th.join }
    end

    def test_emptyrestore
        Puppet::Storage.load
        Puppet::Storage.store
        Puppet::Storage.clear
        Puppet::Storage.load

        f = mkfile()
        state = Puppet::Storage.cache(f)
        assert_same Hash, state.class
        assert_equal 0, state.size
    end
end

# $Id: storage.rb 1793 2006-10-16 22:01:40Z luke $
