/***************************************
  $Header: /home/amb/wwwoffle/RCS/monitor.c 1.19 1998/07/20 18:55:45 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.3.
  A header file for all of the programs wwwoffle, wwwoffled.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1998 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>

#include "wwwoffle.h"
#include "misc.h"
#include "config.h"
#include "errors.h"


static void MonitorFormParse(int fd,char *request_body);


/*++++++++++++++++++++++++++++++++++++++
  Send to the client a page to allow monitoring using HTML.

  int fd The file descriptor of the client.

  URL *Url The URL that was used to request this page.

  char *request_body The body of the HTTP request.
  ++++++++++++++++++++++++++++++++++++++*/

void MonitorPage(int fd,URL *Url,char *request_body)
{
 if(!strcmp("/monitor-options/",Url->path))
   {
    char interval[8];
    sprintf(interval,"%d",MonitorInterval);
    HTMLMessage(fd,200,"WWWOFFLE Monitor Page",NULL,"MonitorPage",
                "url",Url->args?Url->args:"",
                "interval",interval,
                NULL);
   }
 else if(!strcmp("/monitor-request/",Url->path))
    MonitorFormParse(fd,request_body);
 else
   {
    char *page=(char*)malloc(strlen(Url->path)+(Url->args?strlen(Url->args):0)+4);
    if(Url->args)
       sprintf(page,"%s?%s",Url->path,Url->args);
    else
       sprintf(page,"%s",Url->path);

    HTMLMessage(fd,404,"WWWOFFLE Illegal Monitor Page",NULL,"MonitorIllegal",
                "url",page,
                NULL);
    free(page);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Parse the reply from the form.

  int fd The file descriptor of the client.

  char *request_body The body of the HTTP request sent by the browser.
  ++++++++++++++++++++++++++++++++++++++*/

static void MonitorFormParse(int fd,char *request_body)
{
 int i,mfd=-1;
 char *copy,*url=NULL;
 URL *Url;
 int interval=MonitorInterval;

 if(!request_body)
   {
    HTMLMessage(fd,404,"WWWOFFLE Monitor Form Error",NULL,"MonitorFormError",
                "body","",
                NULL);
    return;
   }

 copy=(char*)malloc(strlen(request_body)+1);
 strcpy(copy,request_body);

 for(i=0;copy[i];i++)
   {
    if(i!=0 && copy[i-1]=='&')
       copy[i-1]=0;
    if(i==0 || copy[i-1]==0)
      {
       if(!strncmp("url=",&copy[i],4))
          url=&copy[i+4];
       if(!strncmp("interval=",&copy[i],9))
          interval=atoi(&copy[i+9]);
      }
   }

 if(interval<0)
    interval=0;

 if(url==NULL || *url==0)
   {
    HTMLMessage(fd,404,"WWWOFFLE Monitor Form Error",NULL,"MonitorFormError",
                "body",request_body,
                NULL);
    free(copy);
    return;
   }

 url=URLDecode(url,1);
 Url=SplitURL(url);

 mfd=CreateMonitorSpoolFile(Url,interval);
 if(mfd==-1)
    HTMLMessage(fd,500,"WWWOFFLE Server Error",NULL,"ServerError",
                "error","Cannot open file to store monitor request",
                NULL);
 else
   {
    char intervalstr[8];
    int ofd;
    char *req=RequestURL(Url,NULL);

    write_string(mfd,req);

    close(mfd);

    ofd=OpenOutgoingSpoolFile(0);

    if(ofd==-1)
       PrintMessage(Warning,"Cannot open outgoing spool file for monitored URL '%s' [%!s].",url);
    else
      {
       write_string(ofd,req);

       CloseOutgoingSpoolFile(ofd,Url);
      }

    sprintf(intervalstr,"%d",interval);

    HTMLMessage(fd,200,"WWWOFFLE Monitor Will Get",NULL,"MonitorWillGet",
                "url",Url->name,
                "interval",intervalstr,
                NULL);

    free(req);
   }

 free(copy);

 FreeURL(Url);
}


/*++++++++++++++++++++++++++++++++++++++
  Convert monitor requests into outgoing requests.
  ++++++++++++++++++++++++++++++++++++++*/

void RequestMonitoredPages(void)
{
 DIR *dir;
 struct dirent* ent;

 /* Open the monitor subdirectory. */

 if(chdir("monitor"))
   {PrintMessage(Warning,"Cannot change to directory 'monitor' [%!s]; no files monitored.");return;}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory 'monitor' [%!s]; no files monitored.");chdir("..");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory 'monitor' [%!s]; no files monitored.");closedir(dir);chdir("..");return;}
 ent=readdir(dir);  /* skip .. */

 /* Scan through all of the files. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(lstat(ent->d_name,&buf))
      {PrintMessage(Inform,"Cannot stat file 'monitor/%s' [%!s]; race condition?",ent->d_name);return;}
    else if(S_ISREG(buf.st_mode) && *ent->d_name=='O')
      {
       char *url=FileNameToURL(ent->d_name);
       int interval,last,next;

       chdir("..");

       MonitorTimes(ent->d_name,&interval,&last,&next);

       PrintMessage(Debug,"Monitoring %s; last=%d interval=%d next=%d =>%s",ent->d_name,last,interval,next,
                    (next==0)?"Yes":"No");

       chdir("monitor");

       if(next==0)
         {
          int ifd=open(ent->d_name,O_RDONLY);
          init_buffer(ifd);

          if(ifd==-1)
             PrintMessage(Warning,"Cannot open monitored file 'monitor/%s' to read [%!s].",ent->d_name);
          else
            {
             int ofd;

             chdir("..");

             ofd=OpenOutgoingSpoolFile(0);

             if(ofd==-1)
                PrintMessage(Warning,"Cannot open outgoing spool file for monitored URL '%s' [%!s].",url);
             else
               {
                char *contents=(char*)malloc(buf.st_size);
                URL *Url=SplitURL(url);

                read_data(ifd,contents,buf.st_size);
                write_data(ofd,contents,buf.st_size);

                CloseOutgoingSpoolFile(ofd,Url);

                free(contents);
                FreeURL(Url);
               }

             chdir("monitor");

             close(ifd);
             free(url);
            }
         }
      }
   }

 chdir("..");

 closedir(dir);
}


/*++++++++++++++++++++++++++++++++++++++
  Calculate the monitoring interval and times of previous and next monitoring.

  char *name The name of the file in the monitor directory.

  int *interval The interval in days between monitoring files.

  int *last The time in days ago that the file was last monitored.

  int *next The time in days from now that it should next be monitored.
  ++++++++++++++++++++++++++++++++++++++*/

void MonitorTimes(char *name,int *interval,int *last,int *next)
{
 struct stat buf;
 long now=time(NULL);
 char *file=(char*)malloc(strlen(name)+1);
 int mfd;

 *interval=*last=*next=0;

 if(chdir("monitor"))
    return;

 strcpy(file,name);

 *file='M';
 mfd=open(file,O_RDONLY);
 if(mfd!=-1)
   {
    char str[10];

    strcpy(str,"        ");
    read(mfd,str,8);
    if(sscanf(str,"%d",interval)!=1)
       *interval=MonitorInterval;

    close(mfd);
   }
 else
    *interval=MonitorInterval;

 if(*interval<0)
    *interval=0;

 *file='O';
 if(*interval && !lstat(file,&buf))
   {
    int atimedays=(now-buf.st_atime+3600)/(3600*24);
    int mtimedays=(now-buf.st_mtime+3600)/(3600*24);

    *last=atimedays;
    *next=*interval-(mtimedays%*interval);

    if(*next==*interval && *last)
       *next=0;

    if((*interval-*last)<*next)
       *next=(*interval-*last);

    if(*next<0)
       *next=0;
   }
 else
    *last=*next=0;

 free(file);

 chdir("..");
}
