/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Verwendet: GTK, Imlib                                                     *
 * Modul:     picuti.c                                                       *
 *            Funktionen zur Behandlung von Bilddateien                      *
 * Autor:     Andreas Tille                                                  *
 * Datum:     18.02.1998                                                     *
 * Changes:   19.03.1998: Imlib nur zuer Darstellung, Laden der Dateien      *
 *                        mittels kopiertem load.c aus Imlib                 *
 *                                                                           *
 *****************************************************************************/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/param.h>

#include <gdk_imlib.h>
#include <gdk/gdk.h>

#include "paul.h"

#ifdef __DMALLOC__
#include <dmalloc.h>
#endif

PICTURE *BILD(GList *l) 
/* extracts PICTURE-data from list
 */
{
   if ( !l ) return NULL;
   if (((PICTURE *)(l->data))->id == PAUL_ID ) return (PICTURE *)(l->data);
   g_warning("No PICTURE structur in list!");
   return NULL;
}

int NBILDER(GList *l)
/* Get number of pictures in list
 */
{
   register int n = 0;
   
   if ( !l ) return 0;
   
   do {
      if ( BILD(l) ) n++;
   } while ( (l = l->next) );
   
   return n;
}

void Backup(const char *name)
/* Erzeugt Backup-Datei <name>~
 * --- Parameter: ---
 * char *name     : Pfad zu der Datei
 */
{
   char   bname[MAXPATHLEN];
   struct stat stat_buf;

   if ( !name ) return;   
   if ( stat(name, &stat_buf) ) return; /* name gibt's gar nicht */
   strcat(strcpy(bname, name), "~");
   rename(name, bname);
}

int CheckInBounds(PICTURE *bild, BOX *cut)
/* Testet, ob die Koordinaten bx, by noch im Bild liegen
 * --- Parameter: ---
 * PICTURE *bild           : Bild
 * int     bx              : x-Koordinate
 * int     by              : y-Koordinate
 * --- R"uckgabe: ---
 * int     CheckInBounds() : -1 fr auerhalb, 0 fr genausogro wie Bild,
 *                            1 fr innerhalb
 */
{
   if ( cut->x1 < 0 || cut->x2 < 0 || cut->y1 < 0 || cut->y2 < 0 ) {
      g_warning("Negative coordinates %s (%i*%i)-(%i,%i)!",
                bild->file, cut->x1, cut->y1, cut->x2, cut->y2);
      return -1;
   }
   if ( cut->x1 > cut->x2 ) iswap(cut->x1, cut->x2);
   if ( cut->y1 > cut->y2 ) iswap(cut->y1, cut->y2);
   if ( (cut->x2 > bild->W) || (cut->y2 > bild->H) ) {
      g_warning("Box outside image %s (%i*%i)!", bild->file, bild->W, bild->H);
      return -1;
   }
   if ( cut->x1 == 0       && cut->y1 == 0 &&
        cut->x2 == bild->W && cut->y2 == bild->H ) {
      g_warning("Box equal to image (%i*%i).", cut->x2-cut->x1, cut->y2-cut->y1);
      return 0;
   }
   return 1;
}

void CreateShowPixmaps(GList *piclist)
/* create Imlib structures to given image data or tells imlib that data changed
 * --- Parameter ---
 * GList *piclist : list of images
 * --- Return ---
 * GList *piclist : list of images with valid Imlib data
 */
{
   PICTURE *bild;
   GList   *pl;

   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) {
      if ( !bild->im ) 
         assert((bild->im = 
                at_imlib_set_image_from_data(bild->DATA, bild->W, bild->H)));
      else
         gdk_imlib_changed_image(bild->im);
   }
}

void FreeBild(GList *piclist)
/* gibt allocierten Speicher von Bild-Strukturen wieder frei
 * --- Parameter: ---
 * GList *piclist     : Liste freizugebender Bilder
 */
{
   PICTURE *bild;
   GList   *pl;

   if ( !piclist ) return;
  
   for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) {
      FreeSpecs(bild->spec);
      if ( bild->im ) { 
         gdk_imlib_kill_image(bild->im);
         bild->im   = NULL;
         bild->DATA = NULL;
      } else {
         FREE(bild->DATA);
      }
      FREE(bild->file); 
      FREE(bild->ext);
      FREE(bild->dir);
      FREE(bild->his);
      FREE(bild->physname);
      if ( bild->n_gamma ) FREE(bild->gamma);
      /* if bild->label ist destroyed, which would make sense in case of using it as    *
       * window of operating image, the functions which relay on (bild->label)->parent  *
       * as list_item widget will fail!!!  Leave that out here                          *
       * if ( bild->label   ) gtk_widget_destroy(bild->label);                          */
      FREE(bild);
   }
}

void FreeSpecs(CHUNK *spec)
/* Free chunks for spezifikation
 * --- Parameter: ---
 * CHUNK *spec: chunks to make free
 */
{
   CHUNK *cp;
   
   if ( spec ) {
      for ( cp = spec; cp->key != NULL; cp++ ) 
         FREE(cp->value);
      FREE(spec);
   }
}

int GetListPosition(GList *list, GList *elem)
/* calculates position from elem in list
 */
{
   register int n = 0;
   register GList *l = list;
   
   if ( !l ) return -1;
   while ( l && l != elem ) {
      l = l->next;
      n++;
   }
   if ( !l ) return -1;
   return n;
}

int NewImage(PICTURE *bild, int w_new, int h_new, unsigned char *data)
/* Set new image data
 * --- Parameter: ---
 * PICTURE      *bild       : picture structur
 * int           w_new      : new width
 * int           h_new      : new height
 * unsigned char data       : new image data
 * --- Return: ---
 * PICTURE      *bild       : new picture structur
 * int           NewImage() : 0 if OK
 */
{
   if ( bild->id != PAUL_ID ) return -1;

   if ( bild->im ) {
      gdk_imlib_kill_image(bild->im);
      bild->im   = NULL;  /* make sure, that bild->im is created later */
   } else 
      FREE(bild->DATA);
   bild->DATA = data;
   bild->W    = w_new;
   bild->H    = h_new;
   return 0;
}

BOX *ReadBoxFromString(char *string)
/* read BOX coordinates from string, if neccessary sort to (left,top)-(right,bottom)
 * --- Parameter: ---
 * char *string             : string to read
 * --- R"uckgabe: ---
 * BOX  *ReadBoxFromString(): BOX, NULL if an error occured
 */
{
   int i = 0;
   BOX *cut;
   
   assert ( ( cut = malloc(sizeof(BOX)) ) );
   
   if ( strchr(string, '_' ) )
      i = sscanf(string, "%i,%i_%i,%i", &(cut->x1), &(cut->y1), &(cut->x2), &(cut->y2));
   else if ( strchr(string, '+' ) ) {
      i = sscanf(string, "%ix%i+%i+%i", &(cut->x2), &(cut->y2), &(cut->x1), &(cut->y1));
      cut->x2 += cut->x1;
      cut->y2 += cut->y1;
   } 
   if ( i != 4 ) {
      g_warning("Invalid box format! Use: \"x0,y0_x1,y1\" or \"wxh+x0+y0\"");
      free(cut);
      return NULL;
   }
   if ( cut->x1 < 0 || cut->y1 < 0 || cut->x2 < 0 || cut->y2 < 0 )  {
      g_warning("Box coordinates out of range");
      free(cut);
      return NULL;
   }
   if ( cut->x1 == cut->x2 && cut->y1 == cut->y2 ) {
      g_warning("(left,top) == (right,buttom) ... zero boxes are not supported");
      free(cut);
      return NULL;
   }
   return cut;
}

static int Monochrome(PICTURE *bild)
/* Testet, ob jeweils drei Byte bereinstimmen
 * --- Parameter: ---
 * PICTURE *bild      : vermeintlich monochromes Bild
 * --- R"uckgabe: ---
 * int Monochrome()   : 1 falls Monochrome, sonst 3 (samples per pixel!)
 */
{
   register unsigned char *ap, *fip;
   
   for ( fip = (ap = bild->DATA) + bild->size; ap < fip; ap += 3 )
      if ( (*ap != *(ap+1) || *ap != *(ap+2)) &&
           (*ap || *(ap+2)) )                          /* schon Grnbild ?? */
         return 3;

   return 1;
}
   
static int Gray2Green(unsigned char *data, unsigned int n)
/* Erzeugt aus einem monochromen Bild ein grnes, indem RGB -> 0X0 
 * gesetzt wird
 * ES WIRD NICHT GETESTET, OB DAS BILD TATSCHLICH MONOCHROM IST!
 * --- Parameter: ---
 * unsigned char *data: RGB-Daten des monochromen Bildes
 * unsigned int   n   : Anzahl der RGB-Tripel
 * --- R"uckgabe: ---
 * unsigned char *data: RGB-Daten des monochromen Bildes, 
 *                      wobei R=0 und B=0
 * int Gray2Green()   : 0 fr OK
 */
{
   unsigned char *ap, *fip;
   
   if ( !data ) return -1;
   for ( fip = (ap = data) + 3*n; ap < fip; ap += 3 ) 
      *ap = *(ap+2) = 0;
      
   return 0;
}

char *NewFileName(PICTURE *bild, char *appendix)
/* Creates a new file name from the old one and an appendix
 * deletes old extension if neccessary
 * --- Parameter: ---
 * PICTURE *bild         : picture structure with old name
 * char    *appendix     : Appendix
 * --- R"uckgabe: ---
 * PICTURE *bild         : picture structure with new name
 * char    *NewFileName(): new name
 */
{
   char *filename;

   if ( !IS_PICTURE(bild) ) return NULL;
   if ( !bild->file ) return NULL;
   if ( !appendix   ) appendix = "";
   
   assert ( (filename = malloc(strlen(bild->file) + strlen(appendix) + 1)) != NULL );
   strcpy(filename, bild->file);
   FREE(bild->file);
   FREE(bild->ext);
   return bild->file = strcat(filename, appendix);
}

PICTURE *ReadPic(char *file, OPTIONS *p)
/* liest Bild-Datei
 * --- Parameter: ---
 * char    *file       : filename of image file
 * OPTIONS *p          : flag : what to do 
 *                     : shr  : level of shrinkage
 * --- Rueckgabe: ---
 * PICTURE *ReadPic()  : new PICTURE structure containing contents of file or NULL
 */
{
   PICTURE *bild;
   struct   stat  stat_buf;
   
   if ( stat(file, &stat_buf) ) {
      g_warning("File %s doesnt exist", file);
      return NULL;
   }
   if ( !(bild = InitSpec()) ) return NULL;
   
   if ( (LoadPicture(bild, file, p->f)) == NULL ) {
      g_warning("Unable to load %s.", file);
      return NULL;
   }
   printf("%s loaded %s%s%s%s%s\n", exename, bild->dir ? bild->dir : "", bild->dir ? "/" : "", 
          bild->file, bild->ext ? "." : "", bild->ext ? bild->ext : "");
   bild->im       = NULL;
   if ( OnlyInfo(p->f) ) return bild;
   if ( bild->spp != 1 ) {
      if ( !NotMonoChrom(p->f) && (bild->spp = Monochrome(bild)) == 1 ) {
         bild->spp = 1;
         Gray2Green(bild->DATA, bild->size);
      }
   }
   if ( Shrink(p->f) ) ShrinkPicture(bild, p->shr);
   return bild;
}

GList *ReadPicFiles(int nfiles, char **efiles, OPTIONS *p)
/* Read a list of image files
 * --- Parameter: ---
 * int       nfiles      : number of filenames
 * char    **efiles      : filenames
 * OPTIONS  *p           : f   : Delrand, weitere fr ReadPic()
 *                         cut : gegebenenfalls Box zum Ausschneiden
 *                         shr : Stufe der Verkleinerungen
 * --- Rueckgabe  : ---
 * PICTURE ReadPicFiles(): allocierter Speicher mit Bildern, NULL im Fehlerfall
 */
{
   PICTURE *bild;
   char    **fp, **fip;
   int      i = 0;
   GList   *piclist = NULL;

   for ( fip = (fp = efiles) + nfiles; fp < fip; fp++ ) {
      if ( !(bild = ReadPic(*fp, p)) ) continue;
      piclist = g_list_append(piclist, bild);
      if ( p->cut && !SaveExtrema(p->f ) ) {
         if ( DelBorder(p->f) ) 
	    g_warning("No cutting (-c) of images if option -r (delete border) was given!");
	 else
            if ( CutAreas(g_list_last(piclist), p) ) {
               FreeBild(piclist);
               return NULL;
	    }
      }
      bild++;
      i++;
   }
   if ( !SaveExtrema(p->f) ) FREE(p->cut);
   return piclist;
}



